/*
 * Demonstrates that sizeof() provides misleading results that (when
 * used inside an assignment operator) can result in unexpected data
 * corruption.
 *
 * All materials here are released under terms of the included
 * COPYRIGHT.txt file.
 * Copyright 2025 by James Carlson <carlsonj@workingcode.com>
 */

#include <inttypes.h>
#include <iostream>
#include <string.h>

using namespace std;

/**
 * The SimplePOD class is non-virtual and contains only plain old
 * data.  The initial member is a 64-bit integer and establishes that
 * (on most platforms) the class requires 8-byte alignment.  The rest
 * of the buffers provide 12 bytes of data, which means that the
 * overall size (20 bytes) is NOT 8-byte aligned.  Because sizeof()
 * must provide the array stride, this means that the compiler is
 * compelled to add 4 bytes of extra (hidden) padding and make the
 * overall size 24 bytes.
 */
class SimplePOD {
    int64_t for_alignment;
    char buff1[3];
    char buff2[3];
    char buff3[2];
    char buff4[4];

public:
    size_t actual_data_size() const
    {
	// This is a general form for computing the overall data size,
	// including any hidden padding within the structure, but --
	// crucially -- not including any trailing padding.  Note that
	// just adding the sizes of the members together won't get the
	// correct size because it will miss padding between members.
	// And note that C++ provides no native means to get this
	// "actual length" value.
	return (buff4 - (const char *)this) + sizeof (buff4);
    }

    size_t get_data_size() const
    {
#ifdef FIX_BUG
	return actual_data_size();
#else
	return sizeof (*this);
#endif
    }

    SimplePOD()
    {
	// This is a simple concrete object, and the members inside
	// are just plain old data, so they don't have their own
	// initializers.  We can just initialize the whole thing in
	// bulk.
	size_t clear_size = get_data_size();
	(void) memset(this, 0, clear_size);
    }

    // Removing the assignment overload outright also fixes the
    // problem, but invalidates the intent of the demonstration, which
    // is to show that sizeof is misleading, and that if you must use
    // an assignment overload, you may need to know this.
#ifndef ALTERNATE_FIX
    SimplePOD(const SimplePOD &other)
    {
	// Just copy the contents in bulk.
	size_t copy_size = get_data_size();
	(void) memcpy(this, &other, copy_size);
    }

    SimplePOD &operator=(const SimplePOD &other)
    {
	if (this != &other) {
	    // Just copy the contents in bulk.
	    size_t copy_size = get_data_size();
	    cout << "Doing assignment by copying " << copy_size << " bytes\n";
	    (void) memcpy(this, &other, copy_size);
	}
	return *this;
    }
#endif

    // Accessors suppress compilation complaints about unused private members.
    char *get_buff1() { return buff1; }
    char *get_buff2() { return buff2; }
    char *get_buff3() { return buff3; }
    char *get_buff4() { return buff4; }
    int64_t get_for_alignment() { return for_alignment; }

    void show_offsets() const
    {
	cout << "  Aligned size is " << sizeof(*this) << '\n';
	cout << "  True size is " << actual_data_size() << '\n';
    }
};

class TestClass : public SimplePOD {
    bool flag1;

public:
    TestClass() :
	flag1(true)
    {}

    TestClass(const TestClass &other) :
	SimplePOD(other),
	flag1(other.flag1)
    {}

    void clear_flag()
    {
	flag1 = false;
    }

    TestClass &operator=(const TestClass &other)
    {
	if (this != &other) {
	    // Demonstrate that our flag is set because it was initialized that way.
	    if (flag1)
		cout << "Flag 1 is initially set\n";
	    else
		cout << "Flag 1 is initially UNSET\n";

	    // Do the assignment of just the base class, not this class.
	    *static_cast<SimplePOD *>(this) = static_cast<const SimplePOD &>(other);

	    // Show that this class has had its private member "flag1"
	    // unexpectedly modified.  The base class assignment
	    // operator should not have touched it.
	    if (flag1)
		cout << "Flag 1 is still set\n";
	    else
		cout << "Flag 1 is unexpectedly UNSET\n";

	    // Complete the assignment operation.
	    flag1 = other.flag1;
	}
	return *this;
    }

    void show_offsets() const
    {
	SimplePOD::show_offsets();
	cout << "  Flag 1 is at offset " << (int)((const char *)&flag1 - (const char *)this) << '\n';
    }
};

class TestClass2 : public TestClass {
    bool flag2;

public:
    TestClass2() :
	flag2(true)
    {}

    TestClass2(const TestClass2 &other) :
	TestClass(other),
	flag2(other.flag2)
    {}

    void clear_flag()
    {
	flag2 = false;
	TestClass::clear_flag();
    }

    TestClass2 &operator=(const TestClass2 &other)
    {
	if (this != &other) {
	    // Show that the flag in 'this' is fine at start.
	    if (flag2 != 0)
		cout << "Flag 2 is initially set\n";
	    else
		cout << "Flag 2 is initially UNSET\n";

	    // Assign *ONLY* the base class.  Expect that our added member is untouched.
	    *static_cast<TestClass *>(this) = static_cast<const TestClass &>(other);

	    // Demonstrate that our flag has now been corrupted by base class copy.
	    if (flag2 != 0)
		cout << "Flag 2 is still set\n";
	    else
		cout << "Flag 2 is unexpectedly UNSET\n";

	    // Complete the assignment operation.
	    flag2 = other.flag2;
	}
	return *this;
    }

    void show_offsets() const
    {
	TestClass::show_offsets();
	cout << "  Flag 2 is at offset " << (int)((const char *)&flag2 - (const char *)this) << '\n';
    }
};

int main(int argc, char **argv)
{
    cout << "Testing:\n";
    cout << "  Sizes: POD=" << sizeof(SimplePOD) << " Test=" << sizeof(TestClass) << " Test2=" << sizeof(TestClass2) << '\n';

    // Show offsets of members in leaf class.
    TestClass2 a;
    a.show_offsets();

    // Set up a new source value but with the flags cleared.
    TestClass2 b;
    b.clear_flag();

    // Invoke the assignment operator to demonstrate the accidental memory corruption.
    a = b;
    return 0;
}
